/**
 * @file      xml_util.c
 * @brief     libxml2 support functions
 * @date      2010-03-12
 *
 * @copyright
 * Copyright 2010 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "xml_util.h"
#include "ptrb.h"

/**
 * walk through XML-tree.
 */
void xml_util_walk_element_node(xmlNodePtr node, xml_util_walk_node_callback func, void* data)
{
  xmlNodePtr cur = xmlFirstElementChild(node);
  func(node, data);
  while (cur) {
    xml_util_walk_element_node(cur, func, data);
    cur = xmlNextElementSibling(cur);
  }
}

static void _xml_util_node_name_match_filter(xmlNodePtr node, void* data)
{
  struct {
    ptrb* buf;
    const xmlChar* name;
  }* udata = data;
  voidPtr ptr = node;
  if (0 == strcmp((char *)node->name, (char *)udata->name)) {
    ptrb_append(udata->buf, &ptr, 1);
  }
}

/**
 * DOM getElementsByTagName like function.
 */
xmlNodePtr* xml_util_getElementsByTagName(xmlNodePtr node, const xmlChar* name, size_t* len)
{
  xmlNodePtr* nodes = NULL;
  struct {
    ptrb* buf;
    const xmlChar* name;
  } data;

  if (!(data.buf = ptrb_create(20))) {
    return NULL;
  }
  data.name = name;
  xml_util_walk_element_node(node, _xml_util_node_name_match_filter, &data);
  if (0 < ptrb_size(data.buf)) {
    nodes = malloc(ptrb_size(data.buf) * sizeof(xmlNodePtr) + 1);
    memcpy(nodes, ptrb_ptr(data.buf), ptrb_size(data.buf) * sizeof(xmlNodePtr));
    nodes[ptrb_size(data.buf)] = NULL;
    *len = ptrb_size(data.buf);
  }
  ptrb_free(data.buf);
  return nodes;
}



#if LIBXML_VERSION < 20700
/* **********************************************************************
* Functions not defined in old tree.h.
* **********************************************************************/

xmlNodePtr xmlFirstElementChild(xmlNodePtr node)
{
  xmlNodePtr p = node->children;
  if (!p) return NULL;
  while (p) {
    if (p->type == XML_ELEMENT_NODE) {
      return p;
    }
    p = p->next;
  }
  return NULL;
}

xmlNodePtr xmlNextElementSibling(xmlNodePtr node)
{
  xmlNodePtr p = node;
  while (p->next) {
    p = p->next;
    if (p->type == XML_ELEMENT_NODE) {
      return p;
    }
  }
  return NULL;
}

xmlNodePtr xmlPreviousElementSibling(xmlNodePtr node)
{
  xmlNodePtr p = node;
  while (p->prev) {
    p = p->prev;
    if (p->type == XML_ELEMENT_NODE) {
      return p;
    }
  }
  return NULL;
}
#endif
