/**
 * @file      zbuf.c
 * @brief     Z-buffer for 3D rendering
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "zbuf.h"
#include "float_util.h"

zbuf* zbuf_create(int width, int height, enum ZBUFCondition cond, double near, double far)
{
  zbuf* self = malloc(sizeof(zbuf));
  if (!self) {
    return NULL;
  }
  self->width = width;
  self->height = height;
  self->near = near;
  self->far = far;
  self->condition = cond;
  self->buffer = malloc(sizeof(double)*self->width*self->height);
  if (NULL == self->buffer) {
    free(self);
    return NULL;
  }
  return self;
}

void zbuf_free(zbuf* self)
{
  if (self) {
    free(self->buffer);
    free(self);
  }
}

void zbuf_clear(zbuf* self)
{
  size_t i;
  size_t max = self->width * self->height;
  double* p = self->buffer;
  for (i = 0; i < max; i++) {
    *p = self->far;
    p++;
  }
}

void zbuf_set_range(zbuf* self, double near, double far)
{
  self->near = near;
  self->far = far;
}

double zbuf_z(zbuf* self, int x, int y)
{
  double* p;
  if (x < 0 || self->width <= x) {
    return 0;
  }
  if (y < 0 || self->height <= y) {
    return 0;
  }
  p = self->buffer + self->width * y + x;
  return *p;
}

ZBUFBool zbuf_test(zbuf* self, int x, int y, double z)
{
  double oz = zbuf_z(self, x, y);
  switch (self->condition) {
  case ZBUFConditionLess:
    if (z < oz) return ZBUF_TRUE; break;
  case ZBUFConditionEqualLess:
    if (z <= oz) return ZBUF_TRUE; break;
  case ZBUFConditionGreater:
    if (z > oz) return ZBUF_TRUE; break;
  case ZBUFConditionEqualGreater:
    if (z >= oz) return ZBUF_TRUE; break;
  case ZBUFConditionEqual:
    if (isDoubleAlmostEquals(z,oz)) {
      return ZBUF_TRUE;
    }
    break;
  case ZBUFConditionNotEqual:
    if (!isDoubleAlmostEquals(z,oz)) {
      return ZBUF_TRUE;
    }
    break;

  }
  return ZBUF_FALSE;
}

ZBUFBool zbuf_set(zbuf* self, int x, int y, double z)
{
  ZBUFBool ret = zbuf_test(self, x, y, z);
  double* p;
  if (ret) {
    p = self->buffer + self->width * y + x;
    *p = z;
  }
  return ret;
}
