/**
 * @file      zbuf.h
 * @brief     Z-buffer for 3D rendering
 * @date      2011-03-11
 *
 * @copyright
 * Copyright 2010-2011 Japan Aerospace Exploration Agency
 *
 */

#ifndef _ZBUF_H_
#define _ZBUF_H_

#include <stdint.h>

typedef int ZBUFBool;
#define ZBUF_TRUE   1
#define ZBUF_FALSE  0

enum ZBUFCondition {
  ZBUFConditionLess,
  ZBUFConditionEqualLess,
  ZBUFConditionGreater,
  ZBUFConditionEqualGreater,
  ZBUFConditionEqual,
  ZBUFConditionNotEqual,
};

typedef struct _zbuf {
  double* buffer;
  size_t width;
  size_t height;
  enum ZBUFCondition condition;
  double near;
  double far;
} zbuf;

__BEGIN_DECLS

/**
 * @brief Create a new z-buffer with the specified parameters
 *
 * @param[in]  width  Width of the z-buffer
 * @param[in]  height Height of the z-buffer
 * @param[in]  cond   Condition of the Z-buffer
 * @param[in]  near   Distance from the viewer to the near clipping plane
 * @param[in]  far    Distance from the viewer to the far clipping plane
 *
 * @return zbuf_create() returns the pointer of the allocated memory if success.
 * It returns NULL, otherwise.
 *
 */
zbuf* zbuf_create(int width, int height, enum ZBUFCondition cond, double near, double far);

/**
 * @brief Free the memory of the specified z-buffer
 *
 * @param[in]  self  z-buffer instance
 */
void zbuf_free(zbuf* self);

/**
 * @brief Clear the z-buffer
 * All depth values are set to far.
 *
 * @param[in]  self  z-buffer instance
 */
void zbuf_clear(zbuf* self);

/**
 * @brief Set near and far clipping planes to window coordinates
 *
 * @param[in]  self  z-buffer instance
 * @param[in]  near  distance from the viewer to the near clipping plane
 * @param[in]  far   distance from the viewer to the far clipping plane
 */
void zbuf_set_range(zbuf* self, double near, double far);

/**
 * @brief Get the Z-value at the specified width and height
 *
 * @param[in]  self  z-buffer instance
 * @param[in]  x     x-coordinates of the z-buffer
 * @param[in]  y     y-coordinates of the z-buffer
 */
double zbuf_z(zbuf* self, int x, int y);

/**
 * @brief Test depth comparing the specified depth under the specified condition
 *
 * @param[in]  self  z-buffer instance
 * @param[in]  x     x-coodinates of the z-buffer
 * @param[in]  y     y-coodinates of the z-buffer
 * @param[in]  z     depth of the z-buffer
 * @return zbuf_test() returns true if the specified condition is match,
 * false otherwise.
 */
ZBUFBool zbuf_test(zbuf* self, int x, int y, double z);

/**
 * @brief Set depth if the specified condition is match
 *
 * @param self  zbuf instance.
 * @param[in]  self  z-buffer instance
 * @param[in]  x     x-coodinates of the z-buffer
 * @param[in]  y     y-coodinates of the z-buffer
 * @param[in]  z     depth of the z-buffer
 * @return zbuf_set() returns true if the specified condition is match,
 * false otherwise.
 */
ZBUFBool zbuf_set(zbuf* self, int x, int y, double z);

__END_DECLS

#endif /* _ZBUF_H_ */
